<?php

namespace App\Http\Controllers\Admin;

use App\Models\State;
use App\Models\District;
use App\Models\Registration;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use App\DataTables\RegistrationsDataTable;

class RegistrationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(RegistrationsDataTable $datatable)
    {
        return $datatable->render('admin.registrations.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.registrations.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // Normalize the mobile number to remove unwanted spaces
        if ($request->has('mobile_number')) {
            $request->merge([
                'mobile_number' => preg_replace('/\s+/', ' ', trim($request->input('mobile_number')))
            ]);
        }

        $request->validate([
            'name' => 'required|string|max:255',
            'father_name' => 'required|string|max:255',
            'mother_name' => 'required|string|max:255',
            'village' => 'required|string|max:255',
            'police_station' => 'required|string|max:255',
            'block' => 'required|string|max:255',
            'district_id' => 'required|exists:districts,id',
            'state_id' => 'required|exists:states,id',
            'mobile_number' => [
                'required',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
                'unique:registrations',
            ],
            'email' => 'required|email|unique:registrations|string|max:255',
            'educational_qualifications' => 'required|string|max:255',
            'photo' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'signature' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'date' => 'nullable|date',
            'date_of_birth' => 'required|date',
            'postal_code' => 'nullable|numeric',
        ]);

        $mobile_number = $request->input('mobile_number');
        $registrationNumber = registrationNumber();
        while (Registration::where('reg_no', $registrationNumber)->exists()) {
            $registrationNumber = registrationNumber();
        }


        $data = $request->only([
            'name',
            'father_name',
            'mother_name',
            'village',
            'police_station',
            'block',
            'district_id',
            'state_id',
            'email',
            'educational_qualifications',
            'date',
            'date_of_birth',
            'postal_code',
        ]);
        $data['user_id'] = auth()->user()->id;
        $data['reg_no'] = $registrationNumber;
        $data['mobile_number'] = $mobile_number;
        $data['added_by'] = 'admin';

        if ($request->hasFile('photo')) {
            $data['photo'] = $request->file('photo')->store('Registration/photos', 'public');
        }

        if ($request->hasFile('signature')) {
            $data['signature'] = $request->file('signature')->store('Registration/signatures', 'public');
        }


        Registration::create($data);

        // Redirect to the index route with success message
        return redirect()->route('registrations.index')->with('success', 'Registration created successfully.');
    }




    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $registration = Registration::with('state', 'district')->findOrFail($id);
        return view('admin.registrations.show', compact('registration'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $registration = Registration::with('state.districts', 'district')->findOrFail($id);
        return view('admin.registrations.edit', compact('registration'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        if ($request->has('mobile_number')) {
            $request->merge([
                'mobile_number' => preg_replace('/\s+/', ' ', trim($request->input('mobile_number')))
            ]);
        }
        // return $request;
        $registration = Registration::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'father_name' => 'required|string|max:255',
            'mother_name' => 'required|string|max:255',
            'village' => 'required|string|max:255',
            'police_station' => 'required|string|max:255',
            'block' => 'required|string|max:255',
            'district_id' => 'required|exists:districts,id',
            'state_id' => 'required|exists:states,id',
            'mobile_number' => [
                'required',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
                Rule::unique('registrations', 'mobile_number')->ignore($registration->id),
            ],
            'email' => 'required|email|unique:registrations,email,' . $registration->id . '|string|max:255',
            'educational_qualifications' => 'required|string|max:255',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'signature' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'date' => 'nullable|date',
            'date_of_birth' => 'required|date',
        ]);

        $data = $request->only([
            'name',
            'father_name',
            'mother_name',
            'village',
            'police_station',
            'block',
            'district_id',
            'state_id',
            'mobile_number',
            'email',
            'educational_qualifications',
            'date',
            'date_of_birth',
        ]);
        $data['user_id'] = auth()->user()->id;

        // $mobile_number = str_replace(['-', '_'], '', $request->input('mobile_number'));
        $mobile_number = $request->input('mobile_number');
        $data['mobile_number'] = $mobile_number;

        if ($request->hasFile('photo')) {
            if ($registration->photo) {
                Storage::disk('public')->delete($registration->photo);
            }
            $data['photo'] = $request->file('photo')->store('Registration/photos', 'public');
        } else {
            $data['photo'] = $registration->photo;
        }

        if ($request->hasFile('signature')) {
            if ($registration->signature) {
                Storage::disk('public')->delete($registration->signature);
            }
            $data['signature'] = $request->file('signature')->store('Registration/signatures', 'public');
        } else {
            $data['signature'] = $registration->signature;
        }

        $registration->update($data);

        return redirect()->route('registrations.index')->with('success', 'Registration updated successfully.');
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $registration = Registration::findOrFail($id);
        if ($registration->photo) {
            Storage::disk('public')->delete($registration->photo);
        }
        if ($registration->signature) {
            Storage::disk('public')->delete($registration->signature);
        }
        $registration->delete();

        return response()->json(['success' => 'Registration deleted successfully.']);
        // return redirect()->route('registrations.index')->with('success', 'Registration deleted successfully.');
    }

    public function print(string $id)
    {
        $registration = Registration::find($id);

        return view('admin.registrations.print', compact('registration'));
    }

    public function importCSV(Request $request)
    {
        // Validate CSV file input
        $validator = Validator::make($request->all(), [
            'csv_file' => ['required', 'file', 'mimes:csv,txt', 'max:10240'],
        ]);

        // Handle validation errors
        if ($validator->fails()) {
            return redirect()->back()->with(['error' => $validator->errors()->first()]);
        }

        $totalRecords = 0;
        $updatedRecords = 0;
        $skippedRecords = 0;
        $skippedDueToState = 0;
        $skippedDueToDistrict = 0;
        $skippedMessages = []; // To store specific skipped messages

        // Define the expected header format
        $expectedHeaders = [
            'Name',
            'Father Name',
            'Mother Name',
            'Village',
            'Police Station',
            'State',
            'District',
            'Block',
            'Mobile Number',
            'Email',
            'Date',
            'Educational Qualifications',
            'Date of Birth'
        ];
        $expectedColumnCount = count($expectedHeaders);

        // Check if a CSV file is uploaded
        if ($request->hasFile('csv_file')) {
            $file = $request->file('csv_file');
            $file_path = $file->getRealPath();

            // Open the CSV file for reading
            if (($handle = fopen($file_path, 'r')) !== false) {
                // Read and validate the header row
                $header = fgetcsv($handle);
                if ($header !== $expectedHeaders) {
                    fclose($handle);
                    return redirect()->back()->with(['error' => 'CSV file columns do not match the expected format.']);
                }

                // Loop through each row in the CSV
                while (($data = fgetcsv($handle, 1000, ',')) !== false) {
                    // Skip empty rows
                    if (array_filter($data) == []) {
                        continue;
                    }

                    // Check if the number of columns matches the expected count
                    if (count($data) !== $expectedColumnCount) {
                        $skippedRecords++;
                        $skippedMessages[] = "Skipped record due to incorrect number of columns.";
                        continue;
                    }

                    // Generate a unique registration number
                    $registrationNumber = registrationNumber();
                    while (Registration::where('reg_no', $registrationNumber)->exists()) {
                        $registrationNumber = registrationNumber();
                    }

                    // Map CSV row to registration data
                    $registrationData = [
                        'name' => $data[0],
                        'father_name' => $data[1],
                        'mother_name' => $data[2],
                        'village' => $data[3],
                        'police_station' => $data[4],
                        'state' => $data[5],
                        'district' => $data[6],
                        'block' => $data[7],
                        'mobile_number' => $data[8],
                        'email' => $data[9],
                        'date' => parseDate($data[10]), // The Date column
                        'educational_qualifications' => $data[11],
                        'date_of_birth' => parseDate($data[12]), // The Date of Birth column
                        'reg_no' =>  $registrationNumber,
                        'user_id' => auth()->user()->id,
                    ];

                    // Validate state
                    if (empty($registrationData['state'])) {
                        $skippedDueToState++;
                        $skippedRecords++;
                        $skippedMessages[] = "Skipped record due to missing or invalid state.";
                        continue;
                    }

                    $state = State::where('name', $registrationData['state'])->first();
                    if (!$state) {
                        $skippedDueToState++;
                        $skippedRecords++;
                        $skippedMessages[] = "Skipped record due to invalid state: {$registrationData['state']}.";
                        continue;
                    }

                    // Validate district
                    if (empty($registrationData['district'])) {
                        $skippedDueToDistrict++;
                        $skippedRecords++;
                        $skippedMessages[] = "Skipped record due to missing district.";
                        continue;
                    }

                    $district = District::where('name', $registrationData['district'])->where('state_id', $state->id)->first();
                    if (!$district) {
                        $skippedDueToDistrict++;
                        $skippedRecords++;
                        $skippedMessages[] = "Skipped record due to invalid district: {$registrationData['district']} for state {$registrationData['state']}.";
                        continue;
                    }

                    // Check for existing records by email or mobile number
                    try {
                        $existingRegistration = Registration::where('email', $registrationData['email'])
                            ->orWhere('mobile_number', $registrationData['mobile_number'])
                            ->first();

                        if ($existingRegistration) {
                            // Update only fields except email and mobile number
                            $existingRegistration->update([
                                'name' => $registrationData['name'],
                                'father_name' => $registrationData['father_name'],
                                'mother_name' => $registrationData['mother_name'],
                                'village' => $registrationData['village'],
                                'police_station' => $registrationData['police_station'],
                                'state_id' => $state->id,
                                'district_id' => $district->id,
                                'block' => $registrationData['block'],
                                'date' => $registrationData['date'],
                                'educational_qualifications' => $registrationData['educational_qualifications'],
                                'date_of_birth' => $registrationData['date_of_birth'],
                                'user_id' => $registrationData['user_id'],
                            ]);
                            $updatedRecords++;
                        } else {
                            // Insert a new registration record
                            Registration::create(array_merge($registrationData, [
                                'state_id' => $state->id,
                                'district_id' => $district->id
                            ]));
                            $totalRecords++;
                        }
                    } catch (\Exception $e) {
                        $skippedRecords++;
                        $skippedMessages[] = "Error processing record: {$data[0]} {$data[1]}. Error: {$e->getMessage()}";
                        continue;
                    }
                }
                fclose($handle); // Close the CSV file after processing
            }
        }

        // Return results
        return redirect()->back()->with([
            'success' => "{$totalRecords} records imported successfully. {$updatedRecords} records updated. {$skippedRecords} records skipped. "
                . ($skippedDueToState ? "{$skippedDueToState} records skipped due to invalid state." : '')
                . ($skippedDueToDistrict ? "{$skippedDueToDistrict} records skipped due to invalid district." : '')
                . (count($skippedMessages) ? ' ' . implode(' ', $skippedMessages) : ' ')
        ]);
    }

    public function exportCSV()
    {
        $filename = 'registrations_' . date('Y-m-d') . '.csv';
        $handle = fopen('php://output', 'w');

        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        fputcsv($handle, [
            'Reg No',
            'Name',
            'Father Name',
            'Mother Name',
            'Village',
            'Police Station',
            'State',
            'District',
            'Block',
            'Mobile Number',
            'Email',
            'Date',
            'Educational Qualifications',
            'Date of Birth',
        ]);

        Registration::chunk(1000, function ($registrations) use ($handle) {
            foreach ($registrations as $registration) {
                fputcsv($handle, [
                    $registration->reg_no,
                    $registration->name,
                    $registration->father_name,
                    $registration->mother_name,
                    $registration->village,
                    $registration->police_station,
                    $registration->state->name,
                    $registration->district->name,
                    $registration->block,
                    $registration->mobile_number,
                    $registration->email,
                    $registration->date,
                    $registration->educational_qualifications,
                    $registration->date_of_birth,
                ]);
            }
        });

        fclose($handle);
        exit();
    }
}
