<?php

namespace App\Http\Controllers\Admin;

use App\Models\State;
use App\Models\District;
use Illuminate\Http\Request;
use App\Models\PDCRegistration;
use Illuminate\Validation\Rule;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;
use App\DataTables\PDCRegistrationDataTable;
use Symfony\Component\HttpFoundation\StreamedResponse;


class PDCRegistrationController extends Controller
{
    public function index(PDCRegistrationDataTable $datatable)
    {
        return $datatable->render('admin.pdcr.index');
    }


    public function districtSearch(Request $request)
    {

        $query = PDCRegistration::query();

        // Filter by state
        if ($request->filled('state_id')) {
            $query->where('state_id', $request->state_id);

            // Filter by district
            if ($request->filled('district_id')) {
                $query->where('district_id', $request->district_id);
            }

            // Get filtered results
            $registrations = $query->get();

            return view('admin.pdcr.district_search_index', compact('registrations'));
        }
    }

    public function districtcsv(Request $request)
    {

        $filename = 'pdc_registrations_' . date('Y-m-d') . '.csv';

        // Filter parameters from the request
        $stateId = $request->state_id;
        $districtId = $request->district_id;

        // Use StreamedResponse for better handling
        return new StreamedResponse(function () use ($stateId, $districtId) {
            $handle = fopen('php://output', 'w');

            // Add UTF-8 BOM for Excel compatibility
            fprintf($handle, chr(0xEF) . chr(0xBB) . chr(0xBF));

            // Add CSV header row
            fputcsv($handle, [
                'Reg No',
                'Name',
                'Name (हिन्दी)',
                'Father Name',
                'Father Name (पिता का नाम)',
                'Mother Name',
                'Mother Name (मां का नाम)',
                'Village',
                'Police Station',
                'State',
                'State(राज्य)',
                'District',
                'District(ज़िला)',
                'Block',
                'Mobile Number',
                'Email',
                'Date',
                'Educational Qualifications',
                'Date of Birth',
            ]);

            // Fetch filtered data in chunks
            PDCRegistration::with(['state', 'district']) // Eager load relationships
                ->when($stateId, function ($query) use ($stateId) {
                    $query->where('state_id', $stateId); // Filter by state_id
                })
                ->when($districtId, function ($query) use ($districtId) {
                    $query->where('district_id', $districtId); // Filter by district_id
                })
                ->chunk(1000, function ($registrations) use ($handle) {
                    foreach ($registrations as $registration) {
                        fputcsv($handle, [
                            $registration->reg_no,
                            $registration->name,
                            $registration->hindi_name,
                            $registration->father_name,
                            $registration->father_name_hindi,
                            $registration->mother_name,
                            $registration->mother_name_hindi,
                            $registration->village,
                            $registration->police_station,
                            $registration->state->name ?? '', // Use null coalescing operator
                            $registration->state->name_hindi ?? '',
                            $registration->district->name ?? '',
                            $registration->district->name_hindi ?? '',
                            $registration->block,
                            $registration->mobile_number,
                            $registration->email,
                            $registration->date,
                            $registration->educational_qualifications,
                            $registration->date_of_birth,
                        ]);
                    }
                    flush(); // Free memory
                });

            fclose($handle);
        }, 200, [
            'Content-Type' => 'text/csv; charset=UTF-8',
            'Content-Disposition' => 'attachment; filename="' . $filename . '"',
        ]);
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.pdcr.create');
    }


    public function store(Request $request)
    {
        if ($request->has('mobile_number')) {
            $request->merge([
                'mobile_number' => preg_replace('/\s+/', ' ', trim($request->input('mobile_number')))
            ]);
        }
        // return $request;
        $request->validate([
            'name' => 'required|string|max:255',
            'hindi_name' => 'required|string|max:255',
            'father_name' => 'required|string|max:255',
            'father_name_hindi' => 'required|string|max:255',
            'mother_name' => 'required|string|max:255',
            'mother_name_hindi' => 'required|string|max:255',
            'village' => 'required|string|max:255',
            'police_station' => 'required|string|max:255',
            'block' => 'required|string|max:255',
            'district_id' => 'required|exists:districts,id',
            'state_id' => 'required|exists:states,id',
            'mobile_number' => [
                'required',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
                'unique:pdc_registrations',
            ],
            // 'mobile_number' => 'required|string|max:255|unique:pdc_registrations',
            'email' => 'required|email|unique:pdc_registrations|string|max:255',
            'educational_qualifications' => 'required|string|max:255',
            'photo' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'signature' => 'image|mimes:jpeg,png,jpg,gif|max:2048',
            'date' => 'nullable|date',
            'date_of_birth' => 'required|date',
            'postal_code' => 'nullable|numeric',
            'adhar_card_front' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
            'adhar_card_back' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $registrationNumber = registrationNumber();
        while (PDCRegistration::where('reg_no', $registrationNumber)->orwhere('certificate_no', $registrationNumber)->exists()) {
            $registrationNumber = registrationNumber();
        }

        // $mobile_number = str_replace(['-', '_'], '', $request->input('mobile_number'));
        $mobile_number = $request->input('mobile_number');

        $data = $request->only([
            'name',
            'hindi_name',
            'father_name',
            'father_name_hindi',
            'mother_name',
            'mother_name_hindi',
            'village',
            'police_station',
            'block',
            'district_id',
            'state_id',
            'email',
            'educational_qualifications',
            'date',
            'date_of_birth',
            'postal_code,'
        ]);
        $data['user_id'] = auth()->user()->id;
        $data['reg_no'] = $registrationNumber;
        $data['mobile_number'] = $mobile_number;

        $data['added_by'] = 'admin';


        if ($request->hasFile('photo')) {
            $data['photo'] = $request->file('photo')->store('PDC_Registration/photos', 'public');
        }

        if ($request->hasFile('adhar_card_back')) {
            $originalName = pathinfo($request->file('adhar_card_back')->getClientOriginalName(), PATHINFO_FILENAME);
            $extension = $request->file('adhar_card_back')->getClientOriginalExtension();
            $customName = $originalName . '_back.' . $extension;
            $data['adhar_card_back'] = $request->file('adhar_card_back')->storeAs('PDC_Registration/adhar_card', $customName, 'public');
        }
        if ($request->hasFile('adhar_card_front')) {
            $originalName = pathinfo($request->file('adhar_card_front')->getClientOriginalName(), PATHINFO_FILENAME);
            $extension = $request->file('adhar_card_front')->getClientOriginalExtension();
            $customName = $originalName . '_front.' . $extension;
            $data['adhar_card_front'] = $request->file('adhar_card_front')->storeAs('PDC_Registration/adhar_card', $customName, 'public');
        }

        if ($request->hasFile('signature')) {
            $data['signature'] = $request->file('signature')->store('PDC_Registration/signatures', 'public');
        }

        PDCRegistration::create($data);

        return redirect()->route('pdcr.index')->with('success', 'PDC Registration created successfully.');
    }
    public function show(string $id)
    {
        $registration = PDCRegistration::with('state', 'district')->findOrFail($id);
        return view('admin.pdcr.show', compact('registration'));
    }
    public function edit(string $id)
    {
        $registration = PDCRegistration::with('state.districts', 'district')->findOrFail($id);
        return view('admin.pdcr.edit', compact('registration'));
    }
    public function update(Request $request, string $id)
    {
        if ($request->has('mobile_number')) {
            $request->merge([
                'mobile_number' => preg_replace('/\s+/', ' ', trim($request->input('mobile_number')))
            ]);
        }
        // return $request;
        $registration = PDCRegistration::findOrFail($id);

        $request->validate([
            'name' => 'required|string|max:255',
            'hindi_name' => 'required|string|max:255',
            'father_name' => 'required|string|max:255',
            'father_name_hindi' => 'required|string|max:255',
            'mother_name' => 'required|string|max:255',
            'mother_name_hindi' => 'required|string|max:255',
            'village' => 'required|string|max:255',
            'police_station' => 'required|string|max:255',
            'block' => 'required|string|max:255',
            'district_id' => 'required|exists:districts,id',
            'state_id' => 'required|exists:states,id',
            'mobile_number' => [
                'required',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
                Rule::unique('pdc_registrations', 'mobile_number')->ignore($registration->id),
            ],
            // 'mobile_number' => 'required|string|max:255|unique:pdc_registrations,mobile_number,' . $registration->id,
            'email' => 'required|email|unique:pdc_registrations,email,' . $registration->id . '|string|max:255',
            'educational_qualifications' => 'required|string|max:255',
            'photo' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'signature' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'date' => 'nullable|date',
            'date_of_birth' => 'required|date',
            'postal_code' => 'nullable|numeric',
            'adhar_card_front' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'adhar_card_back' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);

        $data = $request->only([
            'name',
            'hindi_name',
            'father_name',
            'father_name_hindi',
            'mother_name',
            'mother_name_hindi',
            'village',
            'police_station',
            'block',
            'district_id',
            'state_id',
            'email',
            'educational_qualifications',
            'date',
            'date_of_birth',
            'postal_code',
        ]);
        $data['user_id'] = auth()->user()->id;

        // $mobile_number = str_replace(['-', '_'], '', $request->input('mobile_number'));

        $mobile_number = $request->input('mobile_number');
        $data['mobile_number'] = $mobile_number;

        if ($request->hasFile('photo')) {
            if ($registration->photo) {
                Storage::disk('public')->delete($registration->photo);
            }
            $data['photo'] = $request->file('photo')->store('PDC_Registration/photos', 'public');
        } else {
            $data['photo'] = $registration->photo;
        }

        if ($request->hasFile('signature')) {
            if ($registration->signature) {
                Storage::disk('public')->delete($registration->signature);
            }
            $data['signature'] = $request->file('signature')->store('PDC_Registration/signatures', 'public');
        } else {
            $data['signature'] = $registration->signature;
        }

        if ($request->hasFile('adhar_card_back')) {
            if ($registration->adhar_card_back) {
                Storage::disk('public')->delete($registration->adhar_card_back);
            }
            $originalName = pathinfo($request->file('adhar_card_back')->getClientOriginalName(), PATHINFO_FILENAME);
            $extension = $request->file('adhar_card_back')->getClientOriginalExtension();
            $customName = $originalName . '_back.' . $extension;
            $data['adhar_card_back'] = $request->file('adhar_card_back')->storeAs('PDC_Registration/adhar_card', $customName, 'public');
        } else {
            $data['adhar_card_back'] = $registration->adhar_card_back;
        }

        if ($request->hasFile('adhar_card_front')) {
            if ($registration->adhar_card_front) {
                Storage::disk('public')->delete($registration->adhar_card_front);
            }
            $originalName = pathinfo($request->file('adhar_card_front')->getClientOriginalName(), PATHINFO_FILENAME);
            $extension = $request->file('adhar_card_front')->getClientOriginalExtension();
            $customName = $originalName . '_front.' . $extension;
            $data['adhar_card_front'] = $request->file('adhar_card_front')->storeAs('PDC_Registration/adhar_card', $customName, 'public');
        } else {
            $data['adhar_card_front'] = $registration->adhar_card_front;
        }

        $registration->update($data);

        return redirect()->route('pdcr.index')->with('success', 'PDC Registration updated successfully.');
    }
    public function destroy(string $id)
    {
        // return $request;
        $registration = PDCRegistration::findOrFail($id);
        if ($registration->photo) {
            Storage::disk('public')->delete($registration->photo);
        }
        if ($registration->signature) {
            Storage::disk('public')->delete($registration->signature);
        }
        $registration->delete();

        return response()->json(['success' => 'PDCR Registration deleted successfully.']);
        // return redirect()->route('pdcr.index')->with('success', 'Registration deleted successfully.');
    }
    public function print(string $id)
    {
        $registration = PDCRegistration::find($id);

        return view('admin.pdcr.print', compact('registration'));
    }

    public function importCSV(Request $request)
    {
        // Validation for CSV file
        $validator = Validator::make($request->all(), [
            'csv_file' => ['required', 'file', 'mimes:csv,txt', 'max:10240'],
        ]);

        if ($validator->fails()) {
            return redirect()->back()->with(['error' => $validator->errors()->first()]);
        }

        // Initialize counters for statistics
        $totalRecords = 0;
        $updatedRecords = 0;
        $skippedRecords = 0;
        $skippedDueToState = 0;
        $skippedDueToDistrict = 0;
        $skippedMessages = []; // Store specific skipped messages

        // Define the expected header format
        $expectedHeaders = [
            'Name',
            'Name(हिन्दी)',
            'Father Name',
            'Father Name(पिता का नाम)',
            'Mother Name',
            'Mother Name(मां का नाम)',
            'Village',
            'Police Station',
            'State',
            'District',
            'Block',
            'Mobile Number',
            'Email',
            'Date',
            'Educational Qualifications',
            'Date of Birth'
        ];
        $expectedColumnCount = count($expectedHeaders);

        // Check if a CSV file is uploaded
        if ($request->hasFile('csv_file')) {
            $file = $request->file('csv_file');
            $file_path = $file->getRealPath();

            // Open the CSV file for reading
            if (($handle = fopen($file_path, 'r')) !== false) {
                // Read and validate the header row
                $header = fgetcsv($handle);
                if ($header !== $expectedHeaders) {
                    fclose($handle);
                    return redirect()->back()->with(['error' => 'CSV file columns do not match the expected format.']);
                }

                // Loop through each row in the CSV
                while (($data = fgetcsv($handle, 1000, ',')) !== false) {
                    // Skip empty rows
                    if (array_filter($data) == []) {
                        continue;
                    }

                    // Check if the number of columns matches the expected count
                    if (count($data) !== $expectedColumnCount) {
                        $skippedRecords++;
                        $skippedMessages[] = "Skipped record due to incorrect number of columns.";
                        continue;
                    }

                    try {
                        // Generate a unique registration number
                        $registrationNumber = registrationNumber();
                        while (PDCRegistration::where('reg_no', $registrationNumber)->exists()) {
                            $registrationNumber = registrationNumber();
                        }

                        // Map CSV row to registration data
                        $registrationData = [
                            'name' => $data[0],
                            'hindi_name' => $data[1],
                            'father_name' => $data[2],
                            'father_name_hindi' => $data[3],
                            'mother_name' => $data[4],
                            'mother_name_hindi' => $data[5],
                            'village' => $data[6],
                            'police_station' => $data[7],
                            'state' => $data[8],
                            'district' => $data[9],
                            'block' => $data[10],
                            'mobile_number' => $data[11],
                            'email' => $data[12],
                            'date' => parseDate($data[13]), // The Date column
                            'educational_qualifications' => $data[14],
                            'date_of_birth' => parseDate($data[15]), // The Date of Birth column
                            'user_id' => auth()->user()->id,
                            'reg_no' => $registrationNumber,
                            'added_by' => 'admin',
                        ];

                        // Validate state
                        if (empty($registrationData['state'])) {
                            $skippedDueToState++;
                            $skippedRecords++;
                            $skippedMessages[] = "Skipped record due to missing or invalid state.";
                            continue;
                        }

                        // Find state based on the provided state name and its Hindi version
                        $state = State::where('name', $registrationData['state'])->orWhere('name_hindi', $registrationData['state'])->first();
                        if (!$state) {
                            $skippedDueToState++;
                            $skippedRecords++;
                            $skippedMessages[] = "Skipped record due to invalid state: {$registrationData['state']}.";
                            continue;
                        }

                        // Validate district
                        if (empty($registrationData['district'])) {
                            $skippedDueToDistrict++;
                            $skippedRecords++;
                            $skippedMessages[] = "Skipped record due to missing district.";
                            continue;
                        }

                        // Find district based on district name, its Hindi name, and state association
                        $district = District::where('name', $registrationData['district'])
                            ->orWhere('name_hindi', $registrationData['district'])
                            ->where('state_id', $state->id)
                            ->first();

                        if (!$district) {
                            $skippedDueToDistrict++;
                            $skippedRecords++;
                            $skippedMessages[] = "Skipped record due to invalid district: {$registrationData['district']} for state {$registrationData['state']}.";
                            continue;
                        }

                        // Check for existing records by email or mobile number
                        $existingRegistration = PDCRegistration::where('email', $registrationData['email'])
                            ->orWhere('mobile_number', $registrationData['mobile_number'])
                            ->first();

                        if ($existingRegistration) {
                            // Update existing record (exclude email and mobile number)
                            $existingRegistration->update([
                                'name' => $registrationData['name'],
                                'hindi_name' => $registrationData['hindi_name'],
                                'father_name' => $registrationData['father_name'],
                                'father_name_hindi' => $registrationData['father_name_hindi'],
                                'mother_name' => $registrationData['mother_name'],
                                'mother_name_hindi' => $registrationData['mother_name_hindi'],
                                'village' => $registrationData['village'],
                                'police_station' => $registrationData['police_station'],
                                'state_id' => $state->id,
                                'district_id' => $district->id,
                                'block' => $registrationData['block'],
                                'date' => $registrationData['date'],
                                'educational_qualifications' => $registrationData['educational_qualifications'],
                                'date_of_birth' => $registrationData['date_of_birth'],
                                'user_id' => $registrationData['user_id'],
                                'added_by' => 'admin',
                            ]);
                            $updatedRecords++;
                        } else {
                            // Insert a new registration record
                            PDCRegistration::create(array_merge($registrationData, [
                                'state_id' => $state->id,
                                'district_id' => $district->id
                            ]));
                            $totalRecords++;
                        }
                    } catch (\Exception $e) {
                        // Catch any exceptions, log or display them
                        $skippedRecords++;
                        $skippedMessages[] = "Error processing record: {$data[0]} - " . $e->getMessage();
                    }
                }
                fclose($handle); // Close the CSV file after processing
            }
        }

        // Return results with summary
        return redirect()->back()->with([
            'success' => "{$totalRecords} records imported successfully. {$updatedRecords} records updated. {$skippedRecords} records skipped. "
                . ($skippedDueToState ? "{$skippedDueToState} records skipped due to invalid state." : '')
                . ($skippedDueToDistrict ? "{$skippedDueToDistrict} records skipped due to invalid district." : '')
                . (count($skippedMessages) ? ' ' . implode(' ', $skippedMessages) : '')
        ]);
    }

    public function exportCSV()
    {
        $filename = 'pdc_registrations_' . date('Y-m-d') . '.csv';
        $handle = fopen('php://output', 'w');


        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        fprintf($handle, chr(0xEF) . chr(0xBB) . chr(0xBF));

        fputcsv($handle, [
            'Reg No',
            'Name',
            'Name (हिन्दी)',
            'Father Name',
            'Father Name (पिता का नाम)',
            'Mother Name',
            'Mother Name (मां का नाम)',
            'Village',
            'Police Station',
            'State',
            'State(राज्य)',
            'District',
            'District(ज़िला)',
            'Block',
            'Mobile Number',
            'Email',
            'Date',
            'Educational Qualifications',
            'Date of Birth',
        ]);

        PDCRegistration::chunk(1000, function ($registrations) use ($handle) {
            foreach ($registrations as $registration) {
                fputcsv($handle, [
                    $registration->reg_no,
                    $registration->name,
                    $registration->hindi_name,
                    $registration->father_name,
                    $registration->father_name_hindi,
                    $registration->mother_name,
                    $registration->mother_name_hindi,
                    $registration->village,
                    $registration->police_station,
                    $registration->state->name,
                    $registration->state->name_hindi,
                    $registration->district->name,
                    $registration->district->name_hindi,
                    $registration->block,
                    $registration->mobile_number,
                    $registration->email,
                    $registration->date,
                    $registration->educational_qualifications,
                    $registration->date_of_birth,
                ]);
            }
        });

        fclose($handle);
        exit();
    }

    public function generatecertificate(Request $request, $id)
    {
        $registration = PDCRegistration::findOrFail($id);

        if ($registration) {
            $registration->certificate_status = 1;
            $registration->certificate_no = registrationNumber();
            $registration->save();
            return view('Admin.pdcr.certificate', compact('registration', 'request'));
        }
    }

    public function delete(Request $request)
    {
        $registration = PDCRegistration::findOrFail($request->id);
        $registration->delete();
        return response()->json(['success' => 'PDC Registration deleted successfully.']);

    }
}
