<?php

namespace App\Http\Controllers\Admin;

use App\Models\User;
use App\Models\BssDetail;
use Illuminate\Http\Request;
use function Termwind\render;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Hash;
use App\DataTables\BssDetailDataTable;

class BssDetailController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(BssDetailDataTable $datatable)
    {

        return $datatable->render('admin.bss.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.bss.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // return $request;

        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'password' => 'required|string|min:8',
            'registration_no' => 'required|string|max:255',
            'bss_name' => 'required|string',
            'bss_place' => 'required|string',
            'teacher_name' => 'required|string|max:255',
            'teacher_contact_no' => [
                'required',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
            ],
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'status' => 'required|string|in:active,inactive',

        ]);
        if ($validatedData) {
            $user = new User();
            $user->name = $validatedData['name'];
            $user->email = $validatedData['email'];
            $user->password = Hash::make($validatedData['password']);
            $user->type = 'bss';
            $user->active = $validatedData['status'] == 'active' ? 1 : 0;
            $user->email_verified_at = now();

            if ($request->hasFile('avatar')) {
                $avatarPath = $request->file('avatar')->store('avatars', 'public');
                $user->avatar = $avatarPath;
            }

            $user->save();
            $user->assignRole('bss_admin');
            $registrationNumber = registrationNumber('5');
            while (BssDetail::where('serial_no', $registrationNumber)->exists()) {
                $registrationNumber = registrationNumber('5');
            }


            $bssDetail = new BssDetail();
            $bssDetail->user_id = $user->id;
            $bssDetail->registration_no = $validatedData['registration_no'];
            $bssDetail->serial_no = $registrationNumber;
            $bssDetail->bss_name = $validatedData['bss_name'];
            $bssDetail->bss_place = $validatedData['bss_place'];
            $bssDetail->teacher_name = $validatedData['teacher_name'];
            $bssDetail->teacher_contact_no = $validatedData['teacher_contact_no'];
            $bssDetail->save();
            return redirect()->route('bss')->with('success', 'BSS details added successfully.');
        } else {
            return 'error';
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(BssDetail $bssDetail)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $bss = BssDetail::findOrFail($id);
        $user = User::findOrFail($bss->user_id);
        return view('admin.bss.edit', compact('bss', 'user'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        // return $request;

        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable',
            'password' => 'nullable|string|min:8',
            'registration_no' => 'required|string|max:255',
            'bss_name' => 'required|string',
            'bss_place' => 'required|string',
            'teacher_name' => 'required|string|max:255',
            'teacher_contact_no' => [
                'required',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
            ],
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
            'status' => 'required|string|in:active,inactive',
        ]);


        $bss = BssDetail::with('user')->findOrFail($id);


        $bss->user->name = $validatedData['name'];
        // $bss->user->email = $validatedData['email'];
        $bss->user->active = $validatedData['status'] == 'active' ? 1 : 0;

        if (!empty($validatedData['password'])) {
            $validatedData['password'] = bcrypt($validatedData['password']);
        } else {
            unset($validatedData['password']);
        }
        if ($request->hasFile('avatar')) {
            if ($bss->user->avatar && file_exists(storage_path('app/public/' . $bss->user->avatar))) {
                unlink(storage_path('app/public/' . $bss->user->avatar));
            }

            $avatarPath = $request->file('avatar')->store('avatars', 'public');
            $bss->user->avatar = $avatarPath;
        }

        $bss->user->save();
        $bss->user->assignRole('bss_admin');

        $bssDetail = $bss;
        // $bssDetail->serial_no = $registrationNumber;
        $bssDetail->registration_no = $validatedData['registration_no'];
        $bssDetail->bss_name = $validatedData['bss_name'];
        $bssDetail->bss_place = $validatedData['bss_place'];
        $bssDetail->teacher_name = $validatedData['teacher_name'];
        $bssDetail->teacher_contact_no = $validatedData['teacher_contact_no'];

        // Save the updated BssDetail
        $bssDetail->save();

        // Redirect back with a success message
        return redirect()->route('bss')->with('success', 'BSS details updated successfully.');
    }


    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $bss = BssDetail::with('user')->findOrFail($id);
        if ($bss->user->avatar && \Storage::exists($bss->user->avatar)) {
            \Storage::delete($bss->user->avatar);
        }
        $bss->user->delete();
        $bss->delete();
        return response()->json(['success' => 'BSS deleted successfully.']);
    }
}
