<?php

namespace App\Http\Controllers\Admin\Bss;

use App\DataTables\BSSTeacherDataTable;
use App\Http\Controllers\Controller;
use App\Models\BSSTeacher;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class BSSTeacherController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(BSSTeacherDataTable $datatable)
    {
        return $datatable->render('admin.bss.teacher.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create(): \Illuminate\Contracts\View\View
    {
        return view('admin.bss.teacher.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request): \Illuminate\Http\RedirectResponse
    {
        $request->validate([
            'name'      => 'required|string|max:255',
            'email'     => 'required|email|unique:bss_teachers,email',
            'location'  => 'required|string|max:255',
            'phone_no'  => [
                'required',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
                'unique:bss_teachers,phone_no',
            ],
            'dob'       => 'required|date',
        ]);

        $teacher = new BSSTeacher();
        $teacher->bss_id   = Auth::id();
        $teacher->name     = $request->input('name');
        $teacher->email    = $request->input('email');
        $teacher->location = $request->input('location');
        $teacher->phone_no = $request->input('phone_no');
        $teacher->dob      = $request->input('dob');
        $teacher->save();

        return redirect()->route('bss.teachers.index')->with('success', 'Teacher created successfully.');
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(int $id): \Illuminate\Contracts\View\View
    {
        $this->validateBssTeacher($id);

        $teacher = BSSTeacher::findOrFail($id);
        return view('admin.bss.teacher.edit', compact('teacher'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, int $id): \Illuminate\Http\RedirectResponse
    {
        $request->validate([
            'name'      => 'required|string|max:255',
            'email'     => 'required|email|unique:bss_teachers,email,' . $id,
            'location'  => 'required|string|max:255',
            'phone_no'  => [
                'required',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
                'unique:bss_teachers,phone_no,' . $id,
            ],
            'dob'       => 'required|date',
        ]);

        $teacher = BSSTeacher::findOrFail($id);
        $teacher->name     = $request->input('name');
        $teacher->email    = $request->input('email');
        $teacher->location = $request->input('location');
        $teacher->phone_no = $request->input('phone_no');
        $teacher->dob      = $request->input('dob');
        $teacher->save();

        return redirect()->route('bss.teachers.index')->with('success', 'Teacher updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(int $id): \Illuminate\Http\JsonResponse
    {
        $this->validateBssTeacher($id);

        $teacher = BSSTeacher::findOrFail($id);
        $teacher->delete();

        return response()->json(['success' => 'Teacher deleted successfully.']);
    }

    /**
     * Validate that the teacher belongs to the authenticated BSS user.
     */
    private function validateBssTeacher(int $id): void
    {
        // Fetch the teacher by ID
        $teacher = BSSTeacher::find($id);

        // Check if the user has the required permissions
        if (Auth::user()->can('edit_bss_teachers') || Auth::user()->can('delete_bss_teachers')) {
            return;
        }

        // Additional check to ensure the teacher exists and belongs to the authenticated user
        if (!$teacher || $teacher->bss_id !== Auth::id()) {
            abort(403, 'It seems you are not authorized to perform this action.');
        }
    }
}
