<?php

namespace App\Http\Controllers\Admin\Bss;

use App\Models\BSSTeacher;
use Illuminate\Http\Request;
use App\Models\BSSTeacherAttendance;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;

class BSSTeacherAttendanceController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index() {}

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $teachers = BSSTeacher::where('bss_id', Auth::id())
            ->with(['teacher_attandances' => function ($query) {
                $query->whereDate('created_at', now());
            }])
            ->get();

        return view('admin.bss.teacher.attendence.index', compact('teachers'));
    }



    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {

        $data = $request->all();
        $today = now()->toDateString();  // Get today's date in `Y-m-d` format

        foreach ($data['teacher_id'] as $teacherId) {
            // Check if the attendance record already exists for today
            $attendance = BSSTeacherAttendance::where('bss_teacher_id', $teacherId)
                ->whereDate('created_at', $today)
                ->first();

            if (empty($data['review'][$teacherId]) && empty($data['attendance'][$teacherId])) {
                continue;
            }
            if (empty($data['attendance'][$teacherId])) {
                return redirect()->route('bss.teachers.attendence')->with('warning', 'Attendence field is required');
            }
            $attendanceData = [
                'bss_teacher_id' => $teacherId,
                'attendance_status' => $data['attendance'][$teacherId],
                'review' => $data['review'][$teacherId],
                'created_at' => now(),
                'updated_at' => now(),
            ];

            // If record exists, update it, else create a new one
            if ($attendance) {
                $attendance->update($attendanceData);  // Update existing record
            } else {
                BSSTeacherAttendance::create($attendanceData);  // Create new record
            }
        }

        return redirect()->route('bss.teachers.attendence')->with('success', 'Attendance saved successfully');
    }



    /**
     * Display the specified resource.
     */
    public function show(BSSTeacherAttendance $teacherAttendance)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(BSSTeacherAttendance $teacherAttendance)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, BSSTeacherAttendance $teacherAttendance)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(BSSTeacherAttendance $teacherAttendance)
    {
        //
    }


    public function generateTeacherReport(Request $request, $id)
    {

        $teacher = BSSTeacher::findOrFail($id);
        $attendancesQuery = $teacher->teacher_attandances();

        if ($request->has('date_from') || $request->has('date_to') || $request->has('search')) {
            $dateFrom = $request->input('date_from');
            $dateTo = $request->input('date_to');

            if ($dateFrom) {
                $attendancesQuery = $attendancesQuery->whereDate('created_at', '>=', $dateFrom);
            }

            if ($dateTo) {
                $attendancesQuery = $attendancesQuery->whereDate('created_at', '<=', $dateTo);
            }

            if ($request->has('search')) {
                $attendancesQuery = $attendancesQuery->where('attendance_status', 'like', '%' . $request->input('search') . '%');
            }
        }

        $attendances = $attendancesQuery->orderBy('created_at', 'desc')->paginate(1);

        return view('admin.bss.teacher.attendence.report', compact('teacher', 'attendances', 'request'));
    }

    public function generateAllTeacherReport(Request $request)
    {
        // return $request;
        $user = Auth::user();
        if ($user->hasRole('bss_admin'))
            // Fetch teachers with their attendances
            $teachers = BSSTeacher::with(['teacher_attandances' => function ($query) use ($request) {
                // Apply filters to the teacher_attandances relationship
                if ($request->has('date_from') && $request->input('date_from') !== null) {
                    $query->whereDate('created_at', '>=', $request->input('date_from'));
                }

                if ($request->has('date_to') && $request->input('date_to') !== null) {
                    $query->whereDate('created_at', '<=', $request->input('date_to'));
                }

                if ($request->has('search')) {
                    $query->where('attendance_status', 'like', '%' . $request->input('search') . '%');
                }
            }])->where('bss_id', $user->id)->get();
        else {
            $teachers = BSSTeacher::with(['teacher_attandances' => function ($query) use ($request) {
                // Apply filters to the teacher_attandances relationship
                if ($request->has('date_from') && $request->input('date_from') !== null) {
                    $query->whereDate('created_at', '>=', $request->input('date_from'));
                }

                if ($request->has('date_to') && $request->input('date_to') !== null) {
                    $query->whereDate('created_at', '<=', $request->input('date_to'));
                }

                if ($request->has('search')) {
                    $query->where('attendance_status', 'like', '%' . $request->input('search') . '%');
                }
            }])->get();
        }

        // Collect all the attendances
        $attendances = $teachers->flatMap(function ($teacher) {
            return $teacher->teacher_attandances;
        });

        // Paginate the results manually
        $perPage = 20;
        $page = $request->input('page', 1);
        $paginatedAttendances = new \Illuminate\Pagination\LengthAwarePaginator(
            $attendances->forPage($page, $perPage),
            $attendances->count(),
            $perPage,
            $page,
            ['path' => $request->url(), 'query' => $request->query()]
        );

        return view('admin.bss.teacher.attendence.all_teacher_report', [
            'attendances' => $paginatedAttendances,
            'request' => $request,
        ]);
    }

    public function exportCSV($id, Request $request)
    {
        $teacher = BSSTeacher::findOrFail($id);
        $attendancesQuery = $teacher->teacher_attandances();

        // Filter logic based on the request
        if ($request->has('date_from') || $request->has('date_to') || $request->has('search')) {
            $dateFrom = $request->input('date_from');
            $dateTo = $request->input('date_to');

            // Apply date filters
            if ($dateFrom) {
                $attendancesQuery = $attendancesQuery->whereDate('created_at', '>=', $dateFrom);
            }

            if ($dateTo) {
                $attendancesQuery = $attendancesQuery->whereDate('created_at', '<=', $dateTo);
            }

            // Apply search filter for attendance status
            if ($request->has('search')) {
                $attendancesQuery = $attendancesQuery->where('attendance_status', 'like', '%' . $request->input('search') . '%');
            }
        }

        // Prepare the filename
        $filename = 'BSS_Teacher_Attendance_Report_' . date('Y-m-d') . '.csv';
        $handle = fopen('php://output', 'w');

        // Clean the output buffer if needed
        if (ob_get_level() > 0) {
            ob_end_clean();
        }

        // Set headers for CSV download
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        // Add BOM to handle UTF-8 encoding in Excel
        fprintf($handle, chr(0xEF) . chr(0xBB) . chr(0xBF));

        // Output column headers
        fputcsv($handle, [
            'Teacher Name',
            'Attendance Status',
            'Review',
            'Date',
            'Day',
        ]);

        // Fetch filtered attendance records (or all records if no filters)
        $attendances = $attendancesQuery->orderBy('created_at', 'desc')->get();

        // Write the records to the CSV
        foreach ($attendances as $attendance) {
            fputcsv($handle, [
                $attendance->teacher->name,
                $attendance->attendance_status,
                $attendance->review,
                $attendance->created_at->format('Y-m-d'),
                $attendance->created_at->format('l'),
            ]);
        }

        // Close the file handler and exit
        fclose($handle);
        exit();
    }

    public function exportCSVReport(Request $request)
    {
        if (!Auth::user()->hasRole('bss_admin')) {
            $attendancesQuery = BSSTeacherAttendance::query();
        } else {
            $attendancesQuery = BSSTeacherAttendance::whereHas('teacher', function ($query) {
                $query->where('bss_id', Auth::id());
            });
        }

        // Apply filters based on the request
        if ($request->has('date_from') || $request->has('date_to') || $request->has('search')) {
            $dateFrom = $request->input('date_from');
            $dateTo = $request->input('date_to');
            $search = $request->input('search');

            // Date filtering
            if ($dateFrom) {
                $attendancesQuery = $attendancesQuery->whereDate('created_at', '>=', $dateFrom);
            }

            if ($dateTo) {
                $attendancesQuery = $attendancesQuery->whereDate('created_at', '<=', $dateTo);
            }

            // Search filtering
            if ($search) {
                $attendancesQuery = $attendancesQuery->where('attendance_status', 'like', '%' . $search . '%');
            }
        }

        // Fetch all records matching the query
        $attendances = $attendancesQuery->orderBy('created_at', 'desc')->get();

        // Prepare the filename
        $filename = 'BSS_All_Teacher_Attendance_Report_' . date('Y-m-d') . '.csv';

        // Set headers for CSV download before any output
        ob_clean();  // Clean the output buffer to avoid any unwanted content before sending CSV headers
        header('Content-Type: text/csv; charset=UTF-8');
        header('Content-Disposition: attachment; filename="' . $filename . '"');

        // Add BOM for UTF-8 compatibility in Excel
        echo "\xEF\xBB\xBF"; // Write BOM to ensure the file is recognized as UTF-8 in Excel

        // Open the output stream
        $handle = fopen('php://output', 'w');

        // Write the CSV headers
        fputcsv($handle, [
            'Teacher Name',
            'Attendance Status',
            'Review',
            'Date',
            'Day',
        ]);

        // Write the attendance data
        foreach ($attendances as $attendance) {
            fputcsv($handle, [
                $attendance->teacher->name ?? 'N/A', // Handle null teacher name
                $attendance->attendance_status,
                $attendance->review,
                $attendance->created_at->format('Y-m-d'),
                $attendance->created_at->format('l'), // Day of the week
            ]);
        }

        // Close the file handler and exit
        fclose($handle);
        exit(); // Ensure the script stops here and no further output is sent
    }
}
