<?php

namespace App\Http\Controllers\Admin\Bss;

use App\Models\BssStudent;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\DataTables\BSSStudentDataTable;
use Illuminate\Support\Facades\Storage;

class BSSStudentController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(BSSStudentDataTable $datatable)
    {
        return $datatable->render('admin.bss.student.index');
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('admin.bss.student.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // return $request;
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'gender' => 'required|string|in:Male,Female,Other',
            'class' => 'required|string|max:50',
            'category' => 'required|string|max:50',
            'school_name' => 'required|string|max:255',
            'school_address' => 'required|string|max:255',
            'mobile_number' => [
                'required',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
            ],
            'alternative_contact_number' => [
                'nullable',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
            ],
            'dob' => 'required|date',
            'blood_group' => 'required|string|in:A+,A-,B+,B-,AB+,AB-,O+,O-',
            'area_of_residence' => 'required|string|max:255',
            'father_name' => 'required|string|max:255',
            'mother_name' => 'required|string|max:255',
            'mother_profession' => 'required|string|max:255',
            'father_profession' => 'required|string|max:255',
            'years_of_association_with_bss' => 'required|integer|min:0',
            'family_association_with_gayatri_pariwar' => 'required|string|max:1000',
            'photograph' => 'required|image|mimes:jpeg,png,jpg,gif|max:2048', // File validation
        ]);

        if ($request->hasFile('photograph')) {
            $validatedData['photograph'] = $request->file('photograph')->store('uploads', 'public');
        }


        $validatedData['bss_id'] = Auth::id();

        BssStudent::create($validatedData);
        return redirect()->route('bss.students.index')->with('success', 'BSS Student Created Successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $this->validateBssStudent($id);
        $bssStudent = BssStudent::findOrFail($id);
        return view('admin.bss.student.edit', compact('bssStudent'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $validatedData = $request->validate([
            'name' => 'required|string|max:255',
            'gender' => 'required|string|in:Male,Female,Other',
            'class' => 'required|string|max:50',
            'category' => 'required|string|max:50',
            'school_name' => 'required|string|max:255',
            'school_address' => 'required|string|max:255',
            'mobile_number' => [
                'required',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
            ],
            'alternative_contact_number' => [
                'nullable',
                'string',
                'regex:/^\+91\s\d{5}\s\d{5}$/',
            ],
            'dob' => 'required|date',
            'blood_group' => 'required|string|in:A+,A-,B+,B-,AB+,AB-,O+,O-',
            'area_of_residence' => 'required|string|max:255',
            'father_name' => 'required|string|max:255',
            'mother_name' => 'required|string|max:255',
            'mother_profession' => 'required|string|max:255',
            'father_profession' => 'required|string|max:255',
            'years_of_association_with_bss' => 'required|integer|min:0',
            'family_association_with_gayatri_pariwar' => 'required|string|max:1000',
            'photograph' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        ]);
        $bssstudent = BssStudent::findOrFail($id);

        if ($request->hasFile('photograph')) {
            if ($bssstudent->photograph && Storage::exists($bssstudent->photograph)) {
                Storage::delete($bssstudent->photograph);
            }

            $validatedData['photograph']  = $request->file('photograph')->store('uploads', 'public');
        }

        $bssstudent->update($validatedData);

        return redirect()->route('bss.students.index')->with('success', 'BSS Student updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $this->validateBssStudent($id);
        $bssstudent = BssStudent::findOrFail($id);
        $bssstudent->delete();
        return response()->json(['success' => 'BSS Student deleted successfully.']);
    }

    private function validateBssStudent(int $id): void
    {
        // Fetch the teacher by ID
        $teacher = BssStudent::find($id);

        // Check if the user has the required permissions
        if (Auth::user()->can('edit_bss_students') || Auth::user()->can('delete_bss_students')) {
            return;
        }

        // Additional check to ensure the teacher exists and belongs to the authenticated user
        if (!$teacher || $teacher->bss_id !== Auth::id()) {
            abort(403, 'It seems you are not authorized to perform this action.');
        }
    }
}
