<?php

namespace App\DataTables;

use App\Models\Registration;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class RegistrationsDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        $dataTable = new EloquentDataTable($query);

        if ($keyword = request()->get('search')['value']) {
            $dataTable->filter(function ($query) use ($keyword) {
                $query->where(function ($q) use ($keyword) {
                    $q->where('reg_no', 'like', "%{$keyword}%")
                        ->orWhere('name', 'like', "%{$keyword}%")
                        ->orWhere('email', 'like', "%{$keyword}%")
                        ->orWhere('mobile_number', 'like', "%{$keyword}%")
                        ->orWhere('block', 'like', "%{$keyword}%")
                        ->orWhere('district', 'like', "%{$keyword}%")
                        ->orWhere('state', 'like', "%{$keyword}%");
                });
            });
        }

        return $dataTable->addColumn('action', function ($query) {
            return view('admin.registrations.action', [
                'table' => 'registrations-table',
                'model' => collect($query),
                'print_url' => route('registrations.print', $query),
                'edit_url' => route('registrations.edit', $query),
                'view_url' => route('registrations.view', $query),
                'del_url' => route('registrations.destroy', $query),
            ]);
        });
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(Registration $model): QueryBuilder
    {
        return $model->newQuery();
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('registrations-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(0)
            ->selectStyleSingle()
            ->buttons([
                Button::make('csv'),
            ]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('id'),
            Column::make('reg_no')->title('Reg No'),
            Column::make('name')->title('Name'),
            Column::make('email')->title('Email'),
            Column::make('mobile_number')->title('Phone'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'Registrations_' . date('YmdHis');
    }
}
