<?php

namespace App\DataTables;

use App\Models\PDCRegistration;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Button;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Html\Editor\Editor;
use Yajra\DataTables\Html\Editor\Fields;
use Yajra\DataTables\Services\DataTable;

class PDCRegistrationDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        $dataTable = new EloquentDataTable($query);

        if ($keyword = request()->get('search')['value']) {
            $dataTable->filter(function ($query) use ($keyword) {
                $query->where(function ($q) use ($keyword) {
                    $q->where('reg_no', 'like', "%{$keyword}%")
                        ->orWhere('name', 'like', "%{$keyword}%")
                        ->orWhere('email', 'like', "%{$keyword}%")
                        ->orWhere('mobile_number', 'like', "%{$keyword}%")
                        ->orWhere('block', 'like', "%{$keyword}%");

                    $keywordLower = strtolower($keyword);

                    if (strpos($keywordLower, 'generated') !== false && strpos($keywordLower, 'not generated') === false) {
                        $q->orWhere('certificate_status', 1); // Match "generated"
                    }

                    if (strpos($keywordLower, 'not generated') !== false) {
                        $q->orWhere('certificate_status', 0); // Match "not generated"
                    }
                });
            });
        }

        return $dataTable->addColumn('certificate_status', function ($query) {
            if ($query->certificate_status == 1) {
                return '<span class="text-success">Generated</span>';
            } else {
                return '<span class="text-danger">Not Generated</span>';
            }
        })
            ->addColumn('action', function ($query) {
                return view('admin.pdcr.action', [
                    'table' => 'pdcregistration-table',
                    'model' => collect($query),
                    'generate_certificate_with_bg' => route('pdcr.generate.certificate', ['id' => $query->id, 'withbg' => true]),
                    'generate_certificate' => route('pdcr.generate.certificate', ['id' => $query->id]),
                    'print_url' => route('pdcr.print', $query),
                    'edit_url' => route('pdcr.edit', $query),
                    'view_url' => route('pdcr.view', $query),
                    'del_url' => route('pdcr.destroy', $query),
                ]);
            })
            ->rawColumns(['certificate_status', 'action']);
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(PDCRegistration $model): QueryBuilder
    {
        return $model->newQuery();
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('pdcregistration-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(0)
            ->selectStyleSingle();
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        return [
            Column::make('id'),
            Column::make('reg_no')->title('Reg No'),
            Column::make('name')->title('Name'),
            Column::make('email')->title('Email'),
            Column::make('mobile_number')->title('Phone'),
            Column::make('added_by')->title('Added By'),
            Column::computed('certificate_status')->title('Certificate'),
            Column::computed('action')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'PDCRegistration_' . date('YmdHis');
    }
}
