<?php

namespace App\DataTables;

use App\Models\BSSSharedFile;
use Illuminate\Database\Eloquent\Builder as QueryBuilder;
use Illuminate\Support\Facades\Auth;
use Yajra\DataTables\EloquentDataTable;
use Yajra\DataTables\Html\Builder as HtmlBuilder;
use Yajra\DataTables\Html\Column;
use Yajra\DataTables\Services\DataTable;

class BSSSharedFileDataTable extends DataTable
{
    /**
     * Build the DataTable class.
     *
     * @param QueryBuilder $query Results from query() method.
     */
    public function dataTable(QueryBuilder $query): EloquentDataTable
    {
        $dataTable = new EloquentDataTable($query);

        if (Auth::user()->hasRole('super-admin')) {
            $dataTable->addColumn('bss_id', function ($query) {
                return $query->bss->name;
            });
        }

        return $dataTable
            ->addColumn('name', function ($query) {
                $fileUrl = $query->file_path ? asset('storage/SharedFiles/' . $query->file_path) : '#';
                $fileName = htmlspecialchars($query->name ?? 'No Name');
                return '<a href="' . $fileUrl . '" target="_blank">' . $fileName . '</a>';
            })
            ->addColumn('visibility', function ($query) {
                return $query->visibility === 'public'
                    ? '<span class="text-success">Public</span>'
                    : '<span class="text-danger">Private</span>';
            })
            ->addColumn('created_at', function ($query) {
                return date('d-m-Y', strtotime($query->created_at));
            })
            ->addColumn('action', function ($query) {
                return view('admin.bss.shared_files.action', [
                    'table' => 'bsssharedfile-table',
                    'model' => collect($query),
                    'generate_attendence_report' => route('bss.teachers.attendence.report', $query),
                    'del_url' => Auth::user()->hasRole('bss_admin') ? route('bss.shared.files.destroy', $query) : route('shared.files.destroy', $query),
                    'edit_url' => Auth::user()->hasRole('bss_admin') ? route('bss.shared.files.edit', $query) : route('shared.files.edit', $query),
                ]);
            })
            ->rawColumns(['name', 'visibility', 'action']) // Ensure HTML columns are not escaped
            ->setRowId('id');
    }

    /**
     * Get the query source of dataTable.
     */
    public function query(BSSSharedFile $model): QueryBuilder
    {
        $user = Auth::user();

        if ($user->hasRole('bss_admin')) {
            return $model->newQuery()
                ->where('bss_id', $user->id)
                ->orderBy('created_at', 'desc');
        }

        return $model->newQuery()->with('bss')
            ->where('visibility', '!=', 'private')
            ->orderBy('created_at', 'desc');
    }

    /**
     * Optional method if you want to use the html builder.
     */
    public function html(): HtmlBuilder
    {
        return $this->builder()
            ->setTableId('bsssharedfile-table')
            ->columns($this->getColumns())
            ->minifiedAjax()
            ->orderBy(0) // Default order by first column
            ->selectStyleSingle()
            ->buttons([]);
    }

    /**
     * Get the dataTable columns definition.
     */
    public function getColumns(): array
    {
        $columns = [
            Column::make('id')
                ->title('ID')
                ->searchable(true)
                ->orderable(true),
            Column::computed('name')
                ->title('Title')
                ->searchable(true)
                ->orderable(true),
            Column::make('visibility')
                ->title('Visibility')
                ->searchable(true)
                ->orderable(true),
            Column::make('created_at')
                ->title('Shared at')
                ->searchable(true)
                ->orderable(true),
            Column::computed('action')
                ->title('Actions')
                ->exportable(false)
                ->printable(false)
                ->width(60)
                ->addClass('text-center'),
        ];

        if (Auth::user()->hasRole('super-admin')) {
            array_splice($columns, 1, 0, [
                Column::computed('bss_id')
                    ->title('BSS Name')
                    ->searchable(true)
                    ->orderable(true),
            ]);
        }

        return $columns;
    }

    /**
     * Get the filename for export.
     */
    protected function filename(): string
    {
        return 'BSSSharedFile_' . date('YmdHis');
    }
}
